<?php
/**
 * Plugin Name: Core Checker
 * Description: Verifica la integridad del core de WordPress comparando los archivos con los checksums oficiales.
 * Version:     0.1
 * Author:      Ariel Hernández Friz
 */

 if (!defined('ABSPATH')) exit;

 class WPCoreFileScanner {
     private $exclusions = [
         'wp-config.php', // archivo válido que no está en el core
         'robots.txt',
         '.htaccess',
     ];
 
     private $results = [];
     private $summary = [
         'modified' => 0,
         'unknown' => 0,
     ];
 
     public function __construct() {
         add_action('admin_menu', [$this, 'register_menu']);
     }
 
     public function register_menu() {
         add_menu_page(
             'Core Scanner',
             'Core Scanner',
             'manage_options',
             'core-scanner',
             [$this, 'scan_page'],
             'dashicons-shield-alt',
             80
         );
     }
 
     public function scan_page() {
         echo '<div class="wrap"><h1>Escaneo de Archivos del Core</h1>';
 
         $this->scan_core_files();
         $this->render_summary();
         $this->render_scan_results();
 
         echo '</div>';
     }
 
     private function scan_core_files() {
         global $wp_version;
 
         require_once ABSPATH . 'wp-admin/includes/update-core.php';
 
         $core_checksums = get_core_checksums($wp_version, get_locale());
 
         if (!$core_checksums || !is_array($core_checksums)) {
             $this->results[] = ['file' => 'ERROR', 'status' => 'No se pudieron obtener los checksums oficiales.'];
             return;
         }
 
         $iterator = new RecursiveIteratorIterator(
             new RecursiveDirectoryIterator(ABSPATH, RecursiveDirectoryIterator::SKIP_DOTS)
         );
 
         foreach ($iterator as $file) {
             $relative_path = str_replace(ABSPATH, '', $file->getPathname());
 
             // excluir directorios de plugins y themes
             if (strpos($relative_path, 'wp-content/') === 0) {
                 continue;
             }
 
             // excluir archivos definidos como válidos
             if (in_array($relative_path, $this->exclusions)) {
                 continue;
             }
 
             $md5 = md5_file($file->getPathname());
             $status = 'OK';
 
             if (isset($core_checksums[$relative_path])) {
                 if ($md5 !== $core_checksums[$relative_path]) {
                     $status = 'Modificado';
                     $this->summary['modified']++;
                 }
             } else {
                 $status = 'Desconocido';
                 $this->summary['unknown']++;
             }
 
             $mod_time = date("Y-m-d H:i:s", $file->getMTime());
 
             //$modified_by = $this->get_last_editor($file->getPathname());
 
             $this->results[] = [
                 'file' => $relative_path,
                 'status' => $status,
                 'date' => $mod_time,
             //    'editor' => $modified_by,
             ];
         }
     }
 
     private function render_summary() {
         echo '<div style="margin:20px 0;padding:15px;border:1px solid #ccd0d4;background:#f9f9f9;border-radius:6px;">';
         echo '<h2 style="margin-top:0;">Resumen del Escaneo</h2>';
         echo '<p><strong>Archivos Modificados:</strong> ' . $this->summary['modified'] . '</p>';
         echo '<p><strong>Archivos Desconocidos:</strong> ' . $this->summary['unknown'] . '</p>';
         echo '</div>';
     }
 
     private function render_scan_results() {
         echo '<table class="widefat fixed striped">';
         echo '<thead><tr>
                 <th>Archivo</th>
                 <th>Estado</th>
                 <th>Fecha de Modificación</th>
               </tr></thead><tbody>';
 
         foreach ($this->results as $result) {
             $color = $result['status'] === 'OK' ? 'green' : ($result['status'] === 'Modificado' ? 'orange' : 'red');
 
             echo '<tr>';
             echo '<td>' . esc_html($result['file']) . '</td>';
             echo '<td style="color:' . $color . ';"><strong>' . esc_html($result['status']) . '</strong></td>';
             echo '<td>' . esc_html($result['date']) . '</td>';
            
             echo '</tr>';
         }
 
         echo '</tbody></table>';
     }
 
     private function get_last_editor($file_path) {
         // No existe un método nativo en WP para esto,
         // aquí podrías integrar con sistemas de control de versiones (ej: Git)
         // o logs del servidor.
         return 'No disponible';
     }
 }
 
 new WPCoreFileScanner();